using System;

namespace HIPS.Web.Components.Cache
{
    public interface ICacheProvider
    {
        /// <summary>
        /// Gets the value identified by the cache key from the cache.
        /// Returns null if there is no cached value for the specified key.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <returns>The cached value for the specified key, or null if no value found in the cache.</returns>
        T Get<T>(string cacheKey) where T : class;

        /// <summary>
        /// Stores the provided value against the identified cache key using default expiry settings.
        /// If the provided value is null the identified cache key is cleared.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item value to set. If null is provided the key is cleared.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        void Set<T>(string cacheKey, T item, params string[] dependencyKeys) where T : class;

        /// <summary>
        /// Stores the provided value against the identified cache key.
        /// If the provided value is null the identified cache key is cleared.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item value to set. If null is provided the key is cleared.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        void Set<T>(string cacheKey, T item, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys) where T : class;

        /// <summary>
        /// Returns a reference to an object that can be used as an applicable thread lock for this cache for the specified key.
        /// This is useful for funcitonality such as double-check locking. 
        ///  </summary>
        /// <param name="cacheKey">The cache key.</param>
        /// <returns>A cache-specific key-specific thread lock.</returns>
        object GetLock(string cacheKey);

        /// <summary>
        /// Gets the default sliding expiration for new cache values.
        /// The sliding expiration indicates when a cache entry should be evicted if it has not been accessed in a given span of time.
        /// </summary>
        TimeSpan GetDefaultSlidingExpiration();

        /// <summary>
        /// Gets the default absolute expiration for new cache values.
        /// The absolute expiration indicates whether a cache entry should be evicted at a specified date and time.
        /// </summary>
        /// <returns></returns>
        DateTimeOffset GetDefaultAbsoluteExpiration();
    }
}